using System;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This exception is thrown when fault message is received from remote server. 
	/// </summary>
	public class VistALinkFaultException : VistALinkException
	{
		private readonly FaultInfo _faultInfo;
		private readonly bool _fatalFlag;

		/// <summary>
		/// Constructor taking fault info structure containing information about 
		/// remote fault occured. Exceptions created with this exceptions are not fatal
		/// e.g. they do not drop VistALink connection. 
		/// </summary>
		/// <param name="faultInfo">
		///		Fault info structure containing information about 
		///		fault occured at remote server.
		///	</param>
		public VistALinkFaultException( FaultInfo faultInfo ) 
			:this( faultInfo, false ) {}

		/// <summary>
		/// Constructor allowing child exceptions to declare themselves as fatal - 
		/// the ones that drop a connection.
		/// </summary>
		/// <param name="faultInfo">
		///		Fault info structure containing information about 
		///		fault occured at remote server.
		///	</param>
		///	<param name="isFatal">
		///		Flag indicating that an exception is 
		///		fatal, connection was dropped and further execution is impossible without reconnect.
		///	</param>
		protected VistALinkFaultException( FaultInfo faultInfo, bool isFatal ) 
			: base( PrepareExceptionMessageFromFaultInfo( faultInfo ) ) 
		{
			_faultInfo = faultInfo;
			_fatalFlag = isFatal;
		}

		/// <summary>
		/// Formatting method that takes fault info structure and extracts everything from
		/// it into human-readable string.
		/// </summary>
		/// <param name="faultInfo">Fault info structure to describe in string.</param>
		/// <returns>String containing description of fault occured.</returns>
		private static string PrepareExceptionMessageFromFaultInfo( FaultInfo faultInfo )
		{
			return 
				( faultInfo == null ) 
				? SR.Exceptions.FaultExceptionNoFaultInfoAvailable()
				: SR.Exceptions.FaultExceptionFaultMessageReceived( faultInfo.ErrorMessage, faultInfo.ToString() );
		}

		/// <summary>
		/// Fault info structure containing information about the fault occured. 
		/// </summary>
		public FaultInfo FaultInfo
		{
			get
			{
				return _faultInfo;
			}
		}

		/// <summary>
		/// This property indicates if an exception is fatal: server has dropped a connection 
		/// and further communication is impossible.
		/// </summary>
		public bool IsFatal
		{
			get
			{
				return _fatalFlag;
			}
		}
	}
}
